#!/usr/bin/env python
import os
import re
import sys
import shutil
import subprocess

import settings

from PyQt4 import QtGui, QtCore

class HarmonyExporter(QtGui.QMainWindow):
    def __init__(self):
        super(HarmonyExporter, self).__init__()

        self.sceneSelectionList = dict()

        self.setStylesheet()
        self.setWindowTitle("Harmony Scene Exporter v2.2")
        self.setGeometry(300, 300, 620, 650)

        #Status Bar
        self.statusBar = QtGui.QStatusBar()
        self.setStatusBar(self.statusBar)
        self.statusBar.setSizeGripEnabled(False)

        self.buildUI()

    def buildUI(self):

        # ------------------------------------------------------------------------------------------ #
        # Setup Central Widget
        #
        central_widget = QtGui.QWidget()
        self.setCentralWidget(central_widget)

        # ------------------------------------------------------------------------------------------ #
        # Setup Layouts
        #
        central_widget.setLayout(QtGui.QVBoxLayout())
        ejsLayout = QtGui.QHBoxLayout()
        sceneLayout = QtGui.QHBoxLayout()
        baseLayout = QtGui.QHBoxLayout()
        paletteLayout = QtGui.QHBoxLayout()
        exportLayout = QtGui.QHBoxLayout()

        central_widget.layout().addLayout(ejsLayout)
        central_widget.layout().addLayout(sceneLayout)
        central_widget.layout().addLayout(baseLayout)
        central_widget.layout().addLayout(paletteLayout)
        central_widget.layout().addLayout(exportLayout)

        # ------------------------------------------------------------------------------------------ #
        # Add controls
        #
        # Env/Job/Scene
        self.envList = QtGui.QListWidget(self)
        self.envList.setToolTip("Select an Environment")
        self.jobList = QtGui.QListWidget(self)
        self.jobList.setToolTip("Select a Job")
        self.sceneList = QtGui.QListWidget(self)
        self.sceneList.setToolTip("Select Scene(s) to be exported")
        self.sceneList.setSelectionMode(QtGui.QAbstractItemView.ExtendedSelection)
        splitter = QtGui.QSplitter(QtCore.Qt.Horizontal)
        splitter.addWidget(self.envList)
        splitter.addWidget(self.jobList)
        splitter.addWidget(self.sceneList)
        splitter.setStretchFactor(1, 1)
        splitter.setSizes([200, 200, 200])
        splitter.setMinimumHeight(250)
        ejsLayout.addWidget(splitter)

        # Scene
        self.sceneSelectionListTable = QtGui.QTableWidget(self)
        self.sceneSelectionListTable.setColumnCount(4)
        self.sceneSelectionListTable.setHorizontalHeaderLabels(["Export", "Scene Name", "Task Name", "Version"])
        header = self.sceneSelectionListTable.horizontalHeader()
        header.setResizeMode(QtGui.QHeaderView.Stretch)
        add_pushButton = QtGui.QPushButton("Add", self)
        add_pushButton.setToolTip("Add selected scene(s) to the export list")
        add_pushButton.setMinimumWidth(100)
        add_pushButton.setMaximumWidth(100)
        sceneLayout.addWidget(self.sceneSelectionListTable)
        sceneLayout.addWidget(add_pushButton)

        # Basepath
        bp_label = QtGui.QLabel("Basepath:", self)
        self.bp_lineEdit = QtGui.QLineEdit(self)
        self.bp_lineEdit.setToolTip("Basepath location to export scene(s)")
        bp_pushButton = QtGui.QPushButton("Browse...", self)
        bp_pushButton.setToolTip("Browse to select the export location...")
        bp_pushButton.setMinimumWidth(100)
        bp_pushButton.setMaximumWidth(100)
        baseLayout.addWidget(bp_label)
        baseLayout.addWidget(self.bp_lineEdit)
        baseLayout.addWidget(bp_pushButton)

        # Palettes and Compression
        pal_label = QtGui.QLabel("Palettes:", self)
        self.env_checkBox = QtGui.QCheckBox("Env", self)
        self.env_checkBox.setToolTip("Export Environment palette(s)")
        self.env_checkBox.setMinimumWidth(60)
        self.job_checkBox = QtGui.QCheckBox("Job", self)
        self.job_checkBox.setToolTip("Export Job palette(s)")
        self.job_checkBox.setMinimumWidth(60)
        self.job_checkBox.setChecked(1)
        self.scene_checkBox = QtGui.QCheckBox("Scene", self)
        self.scene_checkBox.setToolTip("Export Scene palette(s)")
        self.scene_checkBox.setMinimumWidth(60)
        self.scene_checkBox.setChecked(1)
        horiz_spacer = QtGui.QSpacerItem(50, 0, hPolicy=QtGui.QSizePolicy.Expanding)
        comp_label = QtGui.QLabel("Compression:", self)
        radio_grp = QtGui.QButtonGroup(self)
        self.comp_none_rb = QtGui.QRadioButton("None", self)
        self.comp_none_rb.setToolTip("Do NOT compress exported scene(s)")
        self.comp_zip_rb = QtGui.QRadioButton("Zip", self)
        self.comp_zip_rb.setToolTip("Use zip compression on exported scene(s)")
        self.comp_zip_rb.setChecked(1)
        radio_grp.addButton(self.comp_none_rb)
        radio_grp.addButton(self.comp_zip_rb)
        paletteLayout.addWidget(pal_label)
        paletteLayout.addWidget(self.env_checkBox)
        paletteLayout.addWidget(self.job_checkBox)
        paletteLayout.addWidget(self.scene_checkBox)
        paletteLayout.addItem(horiz_spacer)
        paletteLayout.addWidget(comp_label)
        paletteLayout.addWidget(self.comp_none_rb)
        paletteLayout.addWidget(self.comp_zip_rb)

        # Export
        exp_pushButton = QtGui.QPushButton("Export", self)
        exp_pushButton.setToolTip("Export selected scene(s)")
        exportLayout.addWidget(exp_pushButton)

        # ------------------------------------------------------------------------------------------ #
        # Connect controls
        #
        self.envList.itemClicked.connect(self.refreshJobList)
        self.jobList.itemClicked.connect(self.refreshSceneList)
        add_pushButton.clicked.connect(self.refreshSceneSelectionList)
        bp_pushButton.clicked.connect(self.browseExportPath)
        exp_pushButton.clicked.connect(self.exportSelected)

        self.refreshEnvList()
        self.show()

    def setStylesheet(self):
        qss_file = os.path.join(settings.ROOT_PATH, "stylesheets/PyQt.stylesheet").replace("\\", "/")
        with open(qss_file, "r") as fh:
            self.setStyleSheet(fh.read())

    def refreshEnvList(self):
        self.envList.clear()
        query_cmd = [settings.DBU_APP, "-show", "Name,Env", settings.DB_PATH]
        query_process = subprocess.Popen(query_cmd,
                                         shell=False,
                                         stdout=subprocess.PIPE,
                                         stderr=subprocess.STDOUT)
        output, errors = query_process.communicate()
        groups = output.strip().split('\n')
        result = zip(groups[::2], groups[1::2])
        self.env_jobs = dict()
        for name, env in result:
            name = name.split(":")[-1].strip()
            env = env.split(":")[-1].strip()
            if env not in self.env_jobs:
                self.env_jobs[env] = list()
            if name not in self.env_jobs[env]:
                self.env_jobs[env].append(name)
        self.envList.addItems(self.env_jobs.keys())
        self.envList.sortItems(order=QtCore.Qt.AscendingOrder)

    def refreshJobList(self):
        self.sceneList.clear()
        self.jobList.clear()
        currentEnv = str(self.envList.currentItem().text())
        self.jobList.addItems(self.env_jobs[currentEnv])
        self.jobList.sortItems(order=QtCore.Qt.AscendingOrder)

    def refreshSceneList(self):
        self.sceneList.clear()
        currentJob = str(self.jobList.currentItem().text())
        pat = re.compile(r"^scene-[\d]{3}_[\d]{2}_[\d]{4}$")
        print(os.path.join(settings.BASE_JOB_DIR, currentJob).replace("\\", "/"))
        scenes = os.listdir(os.path.join(settings.BASE_JOB_DIR, currentJob).replace("\\", "/"))
        for scene in scenes:
            print("SCENE : " + scene)
            if not re.search(pat, scene):
                continue
            self.sceneList.addItem(scene.split("-")[-1])
        self.sceneList.sortItems(order=QtCore.Qt.AscendingOrder)

    def refreshSceneSelectionList(self):
        env = str(self.envList.currentItem().text())
        job = str(self.jobList.currentItem().text())
        scenes = self.sceneList.selectedItems()
        for scene in scenes:
            scene = self.validateSceneName(str(scene.text()))
            if scene in self.sceneSelectionList:
                continue
            self.sceneSelectionList[scene] = dict()
            self.sceneSelectionList[scene]["env"] = env
            self.sceneSelectionList[scene]["job"] = job
            self.sceneSelectionList[scene]["scene"] = scene
            self.sceneSelectionList[scene]["data"] = dict()
            self.sceneSelectionList[scene]["data"]["export"] = True
            self.sceneSelectionList[scene]["data"]["scene"] = scene
            self.sceneSelectionList[scene]["data"]["tasks"] = settings.TASK_LIST
            self.sceneSelectionList[scene]["data"]["versions"] = self.getVersions(scene)
            # Populate table
            num_rows = self.sceneSelectionListTable.rowCount()
            self.sceneSelectionListTable.insertRow(num_rows)
            row_widget = QtGui.QWidget()
            row_cb = QtGui.QCheckBox()
            row_cb.setToolTip("Toggle scene export")
            row_cb.setChecked(self.sceneSelectionList[scene]["data"]["export"])
            row_cb_layout = QtGui.QHBoxLayout(row_widget)
            row_cb_layout.addWidget(row_cb)
            row_cb_layout.setAlignment(QtCore.Qt.AlignCenter)
            row_cb_layout.setContentsMargins(0, 0, 0, 0)
            self.sceneSelectionListTable.setCellWidget(num_rows, 0, row_widget)
            row_scene = QtGui.QTableWidgetItem(scene)
            self.sceneSelectionListTable.setItem(num_rows, 1, row_scene)
            row_task_combo = QtGui.QComboBox()
            row_task_combo.setToolTip("Select the task level")
            for item in self.sceneSelectionList[scene]["data"]["tasks"]:
                row_task_combo.addItem(item)
            row_task_combo.setCurrentIndex(1)
            self.sceneSelectionListTable.setCellWidget(num_rows, 2, row_task_combo)
            row_version_combo = QtGui.QComboBox()
            row_version_combo.setToolTip("Select the version")
            for item in self.sceneSelectionList[scene]["data"]["versions"]:
                row_version_combo.addItem(item)
            row_version_combo.setCurrentIndex(row_version_combo.count() - 1)
            self.sceneSelectionListTable.setCellWidget(num_rows, 3, row_version_combo)

    def browseExportPath(self):
        exportPath = QtGui.QFileDialog.getExistingDirectory(self, "Export Location", os.getcwd(), QtGui.QFileDialog.ShowDirsOnly | QtGui.QFileDialog.DontResolveSymlinks)
        self.bp_lineEdit.setText(exportPath)

    def exportSelected(self):
        basepath = str(self.bp_lineEdit.text())
        if not basepath:
            QtGui.QMessageBox.critical(self, "ERROR", "No export location selected!")
            return
        num_rows = self.sceneSelectionListTable.rowCount()
        if num_rows:
            QtGui.QApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
            for row in xrange(0, num_rows):
                row_cb = self.sceneSelectionListTable.cellWidget(row, 0).findChild(QtGui.QCheckBox)
                if not row_cb.isChecked():
                    continue
                dir_name = r"{shotCode}_{task}_{version}"
                proposed_dir_name = dir_name.format(shotCode=self.sceneSelectionListTable.item(row, 1).text(),
                                                    task=self.sceneSelectionListTable.cellWidget(row, 2).currentText(),
                                                    version=self.sceneSelectionListTable.cellWidget(row, 3).currentText())
                out_dir = os.path.join(basepath, proposed_dir_name).replace("\\", "/")
                if not os.path.isdir(out_dir):
                    try:
                        os.makedirs(out_dir)
                    except Exception:
                        QtGui.QApplication.restoreOverrideCursor()
                        QtGui.QMessageBox.critical(self, "ERROR", "Unable to create export location!")
                        return
                shot_code = str(self.sceneSelectionListTable.item(row, 1).text())
                export_cmd = [settings.CC_APP, "-batch", "-export", "-copy", "-resolve_sym", "-keep_broken_sym", "-audio", "-database", "-element", "-timing", "-stage"]
                if self.env_checkBox.isChecked():
                    export_cmd.append("-env_palette")
                if self.job_checkBox.isChecked():
                    export_cmd.append("-job_palette")
                if self.scene_checkBox.isChecked():
                    export_cmd.append("-scene_palette")
                export_cmd.extend(["-export_env", str(self.sceneSelectionList[shot_code]["env"]), "-export_job", str(self.sceneSelectionList[shot_code]["job"]), "-export_scene", str(self.sceneSelectionList[shot_code]["scene"]), "-path", out_dir])
                export_process = subprocess.Popen(export_cmd,
                                                  shell=False,
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.STDOUT)
                output, errors = export_process.communicate()
                if not errors and self.comp_zip_rb.isChecked():
                    shutil.make_archive(os.path.join(basepath, os.path.basename(out_dir)).replace("\\", "/"), 'zip', out_dir)
                    stat_str = "Compressing %s >>> %s" % (os.path.basename(out_dir), os.path.basename(out_dir) + '.zip')
            self.statusBar.showMessage("Export complete!")
            QtGui.QApplication.restoreOverrideCursor()
        else:
            QtGui.QMessageBox.warning(self, "ERROR", "No scenes added to the selection list...")
            return

    def getVersions(self, scene):
        versions = list()
        scene_db_path = os.path.join(settings.BASE_JOB_DIR, str(self.jobList.currentItem().text()), "scene-%s" % scene, "version.db").replace("\\", "/")
        query_cmd = [settings.DBU_APP, "-list", "-show", "ID,Name", "-comp", scene_db_path]
        query_process = subprocess.Popen(query_cmd,
                                         shell=False,
                                         stdout=subprocess.PIPE,
                                         stderr=subprocess.STDOUT)
        output, errors = query_process.communicate()
        output = output.strip().split("\n")
        for item in output:
            versions.append("v%03d" % (int(item.split(",")[0])))
        return versions

    def validateSceneName(self, scene):
        if re.search(re.compile(r"^[\d]{3}_[\d]{2}_[\d]{4}$"), scene):
            return scene
        scene_format = r"{episode}_{sequence}_{shot}"
        job = str(self.jobList.currentItem().text())
        if re.search(re.compile(r"^[a-z]+_[0-9]{3}$"), job):
            return scene_format.format(episode="%03d" % (int(job.split("_")[-1])),
                                       sequence="%02d" % (int(scene.split("_")[0])),
                                       shot="%04d" % (int(scene.split("_")[-1])))
        elif re.search(re.compile(r"^[a-z][0-9]{3}_[0-9]{2}$"), job):
            return scene_format.format(episode="%03d" % (int(job[-6:-3])),
                                       sequence="%02d" % (int(job[-2:])),
                                       shot="%04d" % (int(scene)))


def main():
    ''' Main application '''
    app = QtGui.QApplication(sys.argv)
    app.setStyle("Cleanlooks")
    ex = HarmonyExporter()
    sys.exit(app.exec_())


if __name__ == '__main__':
    main()
