#!/usr/bin/env python
import os
import re
import sys
import subprocess

import settings

from PyQt4 import QtGui, QtCore


class HarmonyRenamer(QtGui.QMainWindow):
    def __init__(self):
        super(HarmonyRenamer, self).__init__()

        self.setStylesheet()
        self.setWindowTitle('Harmony Scene Renamer')
        self.setGeometry(300, 300, 400, 450)

        # Status Bar
        self.statusBar = QtGui.QStatusBar()
        self.setStatusBar(self.statusBar)
        self.statusBar.setSizeGripEnabled(False)

        self.buildUI()

    def buildUI(self):

        # ------------------------------------------------------------------------------------------ #
        # Setup Central Widget
        #
        central_widget = QtGui.QWidget()
        self.setCentralWidget(central_widget)

        # ------------------------------------------------------------------------------------------ #
        # Setup Layouts
        #
        central_widget.setLayout(QtGui.QVBoxLayout())
        ejsLayout = QtGui.QHBoxLayout()
        sceneLayout = QtGui.QHBoxLayout()
        renameLayout = QtGui.QHBoxLayout()

        central_widget.layout().addLayout(ejsLayout)
        central_widget.layout().addLayout(sceneLayout)
        central_widget.layout().addLayout(renameLayout)

        # ------------------------------------------------------------------------------------------ #
        # Add controls
        #
        # Env/Job
        self.envList = QtGui.QListWidget(self)
        self.envList.setToolTip("Select an Environment")
        self.jobList = QtGui.QListWidget(self)
        self.jobList.setToolTip("Select a Job")
        h_splitter = QtGui.QSplitter(QtCore.Qt.Horizontal)
        h_splitter.addWidget(self.envList)
        h_splitter.addWidget(self.jobList)
        h_splitter.setStretchFactor(1, 1)
        h_splitter.setSizes([200, 200])

        # Scene
        self.sceneList = QtGui.QListWidget(self)
        self.sceneList.setToolTip("Select Scene(s) to be renamed")
        self.sceneList.setSelectionMode(QtGui.QAbstractItemView.ExtendedSelection)
        v_splitter = QtGui.QSplitter(QtCore.Qt.Vertical)
        v_splitter.addWidget(h_splitter)
        v_splitter.addWidget(self.sceneList)
        ejsLayout.addWidget(v_splitter)

        # Rename
        rename_pushButton = QtGui.QPushButton("Rename", self)
        rename_pushButton.setToolTip("Rename selected scene(s)")
        renameLayout.addWidget(rename_pushButton)

        # ------------------------------------------------------------------------------------------ #
        # Connect controls
        #
        self.envList.itemClicked.connect(self.refreshJobList)
        self.jobList.itemClicked.connect(self.refreshSceneList)
        rename_pushButton.clicked.connect(self.renameSelected)

        self.refreshEnvList()
        self.show()

    def setStylesheet(self):
        qss_file = os.path.join(settings.ROOT_PATH, "stylesheets/PyQt.stylesheet").replace("\\", "/")
        with open(qss_file, "r") as fh:
            self.setStyleSheet(fh.read())

    def refreshEnvList(self):
        self.envList.clear()
        query_cmd = [settings.DBU_APP, "-show", "Name,Env", settings.DB_PATH]
        query_process = subprocess.Popen(query_cmd,
                                         shell=False,
                                         stdout=subprocess.PIPE,
                                         stderr=subprocess.STDOUT)
        output, errors = query_process.communicate()
        groups = output.strip().split('\n')
        result = zip(groups[::2], groups[1::2])
        self.env_jobs = dict()
        for name, env in result:
            name = name.split(":")[-1]
            env = env.split(":")[-1]
            if env not in self.env_jobs:
                self.env_jobs[env] = list()
            if name not in self.env_jobs[env]:
                self.env_jobs[env].append(name)
        self.envList.addItems(self.env_jobs.keys())
        self.envList.sortItems(order=QtCore.Qt.AscendingOrder)

    def refreshJobList(self):
        self.sceneList.clear()
        self.jobList.clear()
        currentEnv = str(self.envList.currentItem().text())
        self.jobList.addItems(self.env_jobs[currentEnv])
        self.jobList.sortItems(order=QtCore.Qt.AscendingOrder)

    def refreshSceneList(self):
        self.sceneList.clear()
        currentJob = str(self.jobList.currentItem().text())
        pat1 = re.compile(r"^scene-[\d]{3}_[\d]{2}_[\d]{4}$")
        pat2 = re.compile(r"^scene-[\d]{2}_[\d]{3,4}$")
        pat3 = re.compile(r"^scene-[\d]{3,4}$")
        scenes = os.listdir(os.path.join(settings.BASE_JOB_DIR, currentJob).replace("\\", "/").rstrip("\r"))
        for scene in scenes:
            if not re.search(pat1, scene) and not re.search(pat2, scene) and not re.search(pat3, scene):
                continue
            self.sceneList.addItem(scene.split("-")[-1])
        self.sceneList.sortItems(order=QtCore.Qt.AscendingOrder)

    def renameSelected(self):
        scenes = self.sceneList.selectedItems()
        if scenes:
            QtGui.QApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
            for scene in scenes:
                rename_cmd = [settings.CC_APP, "-rename", "-env", str(self.envList.currentItem().text()), "-job", str(self.jobList.currentItem().text()), "-scene", str(scene.text()), "-name", self.validateSceneName(str(scene.text()))]
                rename_process = subprocess.Popen(rename_cmd,
                                                  shell=False,
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.STDOUT)
                output, errors = rename_process.communicate()
            self.refreshSceneList()
            self.statusBar.showMessage("Export complete!")
            QtGui.QApplication.restoreOverrideCursor()
        else:
            QtGui.QMessageBox.warning(self, "ERROR", "No scenes added to the rename list...")

    def validateSceneName(self, scene):
        if re.search(re.compile(r"^[\d]{3}_[\d]{2}_[\d]{4}$"), scene):
            return scene
        scene_format = r"{episode}_{sequence}_{shot}"
        job = str(self.jobList.currentItem().text())
        if re.search(re.compile(r"^[a-z]+_[0-9]{3}$"), job):
            return scene_format.format(episode="%03d" % (int(job.split("_")[-1])),
                                       sequence="%02d" % (int(scene.split("_")[0])),
                                       shot="%04d" % (int(scene.split("_")[-1])))
        elif re.search(re.compile(r"^[a-z][0-9]{3}_[0-9]{2}$"), job):
            return scene_format.format(episode="%03d" % (int(job[-6:-3])),
                                       sequence="%02d" % (int(job[-2:])),
                                       shot="%04d" % (int(scene)))


def main():
    ''' Main application '''
    app = QtGui.QApplication(sys.argv)
    app.setStyle("Cleanlooks")
    ex = HarmonyRenamer()
    sys.exit(app.exec_())


if __name__ == '__main__':
    main()
